using System;
using System.Data;
using System.Reflection;
using gov.va.med.vbecs.DAL;
using gov.va.med.vbecs.ExceptionManagement;
using System.Collections;
using Microsoft.Reporting.WinForms;
using System.IO;
using TABLE = gov.va.med.vbecs.Common.VbecsTables;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;

namespace gov.va.med.vbecs.BOL
{
	/// <summary>
	/// Summary description for TestingWorklist.
	/// </summary>
	public class TestingWorklist:VbecsReport
	{
		private BOL.Patient _patient;
		private ArrayList _specimens;
		//
		private DataTable _patientTestingWorklist;
		private DataTable _unitTestingWorklist;
		private DataTable _rackTestingWorklist;
		private DataTable _reagentTestingWorklist;

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/11/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5304"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>TestingWorklist object initialized</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5305"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Blank Testing Worklist constructor
		/// </summary>
		public TestingWorklist()
		{
			this.IncludePatientTesting = false;
			this.IncludeUnitTesting = false;
			this.IncludeRackTesting = false;
			this.IncludeReagentTesting = false;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/11/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5306"> 
		///		<ExpectedInput>Valid patient object and array list of patient specimen objects</ExpectedInput>
		///		<ExpectedOutput>TestingWorklist object initialized with patient and specimen properties set</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5307"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Testing Worklist constructor for individual patient report
		/// </summary>
		/// <param name="patient">Patient</param>
		/// <param name="specimens">Specimens</param>
		public TestingWorklist(BOL.Patient patient, ArrayList specimens)
		{
			this._patient = patient;
			this._specimens = specimens;
			this.IncludePatientTesting = true;
			this.IncludeUnitTesting = false;
			this.IncludeRackTesting = false;
			this.IncludeReagentTesting = false;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/11/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5308"> 
		///		<ExpectedInput>Valid TestingWorklist object</ExpectedInput>
		///		<ExpectedOutput>IncludePatientTesting indicator property</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="5309"> 
		///		<ExpectedInput>IncludePatientTesting indicator of true</ExpectedInput>
		///		<ExpectedOutput>Updated TestingWorklist object</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="5336"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets value indicating whether patient testing section should be included in Testing Worklist report
		/// </summary>
		public bool IncludePatientTesting
		{
			get
			{
				if (this.ParameterRow.IsNull(ARTIFICIAL.IncludePatientTesting))
				{
					return false;
				}

				return Convert.ToBoolean(this.ParameterRow[ARTIFICIAL.IncludePatientTesting]);
			}
			set
			{
				this.ParameterRow[ARTIFICIAL.IncludePatientTesting] = value;
				//
				RuleBroken("ReportSections", ((this.IncludePatientTesting || this.IncludeRackTesting || this.IncludeReagentTesting || this.IncludeUnitTesting)== false));
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/11/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5310"> 
		///		<ExpectedInput>Valid TestingWorklist object</ExpectedInput>
		///		<ExpectedOutput>IncludeUnitTesting indicator property</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="5311"> 
		///		<ExpectedInput>IncludeUnitTesting indicator of true</ExpectedInput>
		///		<ExpectedOutput>Updated TestingWorklist object</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="5337"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets value indicating whether unit testing section should be included in Testing Worklist report
		/// </summary>
		public bool IncludeUnitTesting
		{
			get
			{
				if (this.ParameterRow.IsNull(ARTIFICIAL.IncludeUnitTesting))
				{
					return false;
				}

				return Convert.ToBoolean(this.ParameterRow[ARTIFICIAL.IncludeUnitTesting]);
			}
			set
			{
				this.ParameterRow[ARTIFICIAL.IncludeUnitTesting] = value;
				//
				RuleBroken("ReportSections", ((this.IncludePatientTesting || this.IncludeRackTesting || this.IncludeReagentTesting || this.IncludeUnitTesting)== false));
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/11/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5312"> 
		///		<ExpectedInput>Valid TestingWorklist object</ExpectedInput>
		///		<ExpectedOutput>IncludeRackTesting indicator property</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5338"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets value indicating whether rack testing section should be included in Testing Worklist report
		/// </summary>
		public bool IncludeRackTesting
		{
			get
			{
				if (this.ParameterRow.IsNull(ARTIFICIAL.IncludeRackTesting))
				{
					return false;
				}
				return Convert.ToBoolean(this.ParameterRow[ARTIFICIAL.IncludeRackTesting]);
			}
			set
			{
				this.ParameterRow[ARTIFICIAL.IncludeRackTesting] = value;
				//
				RuleBroken("ReportSections", ((this.IncludePatientTesting || this.IncludeRackTesting || this.IncludeReagentTesting || this.IncludeUnitTesting)== false));
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/11/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5313"> 
		///		<ExpectedInput>Valid TestingWorklist object</ExpectedInput>
		///		<ExpectedOutput>IncludeReagentTesting indicator property</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="5314"> 
		///		<ExpectedInput>IncludeReagentTesting indicator of true</ExpectedInput>
		///		<ExpectedOutput>Updated TestingWorklist object</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="5339"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets value indicating whether reagent testing section should be included in Testing Worklist report
		/// </summary>
		public bool IncludeReagentTesting
		{
			get
			{
				if (this.ParameterRow.IsNull(ARTIFICIAL.IncludeReagentTesting))
				{
					return false;
				}

				return Convert.ToBoolean(this.ParameterRow[ARTIFICIAL.IncludeReagentTesting]);
			}
			set
			{
				this.ParameterRow[ARTIFICIAL.IncludeReagentTesting] = value;
				//
				RuleBroken("ReportSections", ((this.IncludePatientTesting || this.IncludeRackTesting || this.IncludeReagentTesting || this.IncludeUnitTesting)== false));
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/11/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5316"> 
		///		<ExpectedInput>Valid TestingWorklist object</ExpectedInput>
		///		<ExpectedOutput>Patient property</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="5317"> 
		///		<ExpectedInput>Valid patient object</ExpectedInput>
		///		<ExpectedOutput>Updated TestingWorklist object</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="5340"> 
		///		<ExpectedInput>Patient object with invalid unique identifier</ExpectedInput>
		///		<ExpectedOutput>Business object exception thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets patient for individual patient report
		/// </summary>
		public BOL.Patient Patient
		{
			get
			{
				return this._patient;
			}
			set
			{
				if ((value != null) && (value.PatientGuid == Guid.Empty))
				{
					throw new BOL.BusinessObjectException(Common.StrRes.SysErrMsg.Common.InvalidArgument("patient").ResString);
				}
				//
				this._patient = value;
				//
				if (this._patient != null)
				{
					this.IncludePatientTesting = true;
					this.IncludeUnitTesting = false;
					this.IncludeRackTesting = false;
					this.IncludeReagentTesting = false;
				}
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/11/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5318"> 
		///		<ExpectedInput>Valid TestingWorklist object</ExpectedInput>
		///		<ExpectedOutput>Specimens property</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="5319"> 
		///		<ExpectedInput>Valid array list of specimen objects</ExpectedInput>
		///		<ExpectedOutput>Updated TestingWorklist object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5341"> 
		///		<ExpectedInput>Array list not containing specimen objects</ExpectedInput>
		///		<ExpectedOutput>Business object exception thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Specimens
		/// </summary>
		public ArrayList Specimens
		{
			get
			{
				return this._specimens;
			}
			set
			{
				if ((value != null) && (value.Count>0))
				{
					foreach(object obj in value)
					{
						if (obj.GetType() != typeof(BOL.Specimen))
						{
							throw new BOL.BusinessObjectException("Array list must contain specimen objects");
						}
					}
				}
				this._specimens = value;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/11/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5320"> 
		///		<ExpectedInput>Valid TestingWorklist object</ExpectedInput>
		///		<ExpectedOutput>ReportCriteria property</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5321"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets report criteria text
		/// </summary>
		public string ReportCriteria
		{
			get
			{
				string reportCriteria = "";
				if (this.Patient == null)
				{
					reportCriteria += "Date Range: ";
					reportCriteria += Common.VBECSDateTime.FormatDateTimeString(this.StartDate);
					reportCriteria += " - ";
					reportCriteria += Common.VBECSDateTime.FormatDateTimeString(this.EndDate);
				}
				else
				{
					reportCriteria += "Patient: ";
					reportCriteria += this._patient.DisplayName + "\n";
					reportCriteria += "Specimens: ";
					foreach(BOL.Specimen specimen in this._specimens)
					{
						reportCriteria += specimen.SpecimenUid + ", ";
					}
					if (reportCriteria.EndsWith(", "))
					{
						reportCriteria = reportCriteria.Substring(0,reportCriteria.Length-1);
					}
				}
				//
				return reportCriteria;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/11/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5322"> 
		///		<ExpectedInput>TestingWorklist object with IncludePatientTesting flag set to false</ExpectedInput>
		///		<ExpectedOutput>PatientTestingWorklist property</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5323"> 
		///		<ExpectedInput>TestingWorklist object with IncludePatientTesting flag set to false</ExpectedInput>
		///		<ExpectedOutput>Business object exception thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets patient testing worklist report data
		/// </summary>
		public DataTable PatientTestingWorklist
		{
			get
			{
				if (!this.IncludePatientTesting)
				{
					throw new BOL.BusinessObjectException("Patient testing worklist not included in report");
				}
				else if (this.IncludePatientTesting && (this._patientTestingWorklist==null))
				{
					this.GetPatientTestingWorklist();
				}
				return this._patientTestingWorklist;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/11/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5324"> 
		///		<ExpectedInput>TestingWorklist object with IncludeUnitTesting flag set to true</ExpectedInput>
		///		<ExpectedOutput>UnitTestingWorklist property</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5325"> 
		///		<ExpectedInput>TestingWorklist object with IncludeUnitTesting flag set to false</ExpectedInput>
		///		<ExpectedOutput>Business object exception thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets unit testing worklist report data
		/// </summary>
		public DataTable UnitTestingWorklist
		{
			get
			{
				if (!this.IncludeUnitTesting)
				{
					throw new BOL.BusinessObjectException("unit testing worklist not included in report");
				}
				else if (this.IncludeUnitTesting && (this._unitTestingWorklist==null))
				{
					this.GetBloodUnitTestingWorklist();
				}
				return this._unitTestingWorklist;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/11/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5326"> 
		///		<ExpectedInput>TestingWorklist object with IncludeRackTesting flag set to true</ExpectedInput>
		///		<ExpectedOutput>RackTestingWorklist property</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5327"> 
		///		<ExpectedInput>TestingWorklist object with IncludeRackTesting flag set to false</ExpectedInput>
		///		<ExpectedOutput>Business object exception thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets rack testing worklist report data
		/// </summary>
		public DataTable RackTestingWorklist
		{
			get
			{
				if (!this.IncludeRackTesting)
				{
					throw new BOL.BusinessObjectException("rack testing worklist not included in report");
				}
				else if (this.IncludeRackTesting && (this._rackTestingWorklist==null))
				{
					this.GetRackQCTestingWorklist();
				}
				return this._rackTestingWorklist;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/11/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5328"> 
		///		<ExpectedInput>TestingWorklist object with IncludeReagentTesting flag set to true</ExpectedInput>
		///		<ExpectedOutput>ReagentTestingWorklist property</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5329"> 
		///		<ExpectedInput>TestingWorklist object with IncludeReagentTesting flag set to false</ExpectedInput>
		///		<ExpectedOutput>Business object exception thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets reagent testing worklist report data
		/// </summary>
		public DataTable ReagentTestingWorklist
		{
			get
			{
				if (!this.IncludeReagentTesting)
				{
					throw new BOL.BusinessObjectException("reagent testing worklist not included in report");
				}
				else if (this.IncludeReagentTesting && (this._reagentTestingWorklist==null))
				{
					this.GetReagentQCTestingWorklist();
				}
				return this._reagentTestingWorklist;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/11/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5330"> 
		///		<ExpectedInput>Valid TestingWorklist object</ExpectedInput>
		///		<ExpectedOutput>ReportSectionsTable property</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5331"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets reagent testing worklist report section details
		/// </summary>
		public DataTable ReportSectionsTable
		{
			get
			{
				DataTable dt = new DataTable("Sections");
				dt.Columns.Add(ARTIFICIAL.IncludePatientTesting,typeof(bool));
				dt.Columns.Add(ARTIFICIAL.PatientTestingRowCount,typeof(int));
				dt.Columns.Add(ARTIFICIAL.IncludeUnitTesting,typeof(bool));
				dt.Columns.Add(ARTIFICIAL.UnitTestingRowCount,typeof(int));
				dt.Columns.Add(ARTIFICIAL.IncludeRackTesting,typeof(bool));
				dt.Columns.Add(ARTIFICIAL.RackTestingRowCount,typeof(int));
				dt.Columns.Add(ARTIFICIAL.IncludeReagentTesting,typeof(bool));
				dt.Columns.Add(ARTIFICIAL.ReagentTestingRowCount,typeof(int));
				//
				DataRow drSection = dt.NewRow();
				//
				if (this.IncludePatientTesting)
				{
					drSection[ARTIFICIAL.IncludePatientTesting] = true;
					drSection[ARTIFICIAL.PatientTestingRowCount] = 0;
					foreach(DataRow dr in this._patientTestingWorklist.Rows)
					{
						if ((bool)dr[ARTIFICIAL.NoDataIndicator] == false)
						{
							drSection[ARTIFICIAL.PatientTestingRowCount]=(int)drSection[ARTIFICIAL.PatientTestingRowCount]+1;
						}
					}
				}
				else
				{
					drSection[ARTIFICIAL.IncludePatientTesting] = false;
					drSection[ARTIFICIAL.PatientTestingRowCount] = 0;
				}
				//
				if (this.IncludeUnitTesting)
				{
					drSection[ARTIFICIAL.IncludeUnitTesting] = true;
					drSection[ARTIFICIAL.UnitTestingRowCount] = _unitTestingWorklist.Rows.Count;
				}
				else
				{
					drSection[ARTIFICIAL.IncludeUnitTesting] = false;
					drSection[ARTIFICIAL.UnitTestingRowCount] = 0;
				}
				//
				if (this.IncludeRackTesting)
				{
					drSection[ARTIFICIAL.IncludeRackTesting] = true;
					drSection[ARTIFICIAL.RackTestingRowCount] = _rackTestingWorklist.Rows.Count;
				}
				else
				{
					drSection[ARTIFICIAL.IncludeRackTesting] = false;
					drSection[ARTIFICIAL.RackTestingRowCount] = 0;
				}
				//
				if (this.IncludeReagentTesting)
				{
					drSection[ARTIFICIAL.IncludeReagentTesting] = true;
					drSection[ARTIFICIAL.ReagentTestingRowCount] = _reagentTestingWorklist.Rows.Count;
				}
				else
				{
					drSection[ARTIFICIAL.IncludeReagentTesting] = false;
					drSection[ARTIFICIAL.ReagentTestingRowCount] = 0;
				}
				dt.Rows.Add(drSection);
				//
				return dt;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/11/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5332"> 
		///		<ExpectedInput>Valid TestingWorklist object with all report section flags turned on</ExpectedInput>
		///		<ExpectedOutput>Record count > 0</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5333"> 
		///		<ExpectedInput>Invalid TestingWorklist object with no report section flags turned on</ExpectedInput>
		///		<ExpectedOutput>Record count of zero</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GetTestingWorklistCount
		/// </summary>
		/// <returns>Record count</returns>
		public int GetTestingWorklistCount()
		{
			return DAL.TestingWorklist.GetTestingWorklistCount(this.StartDate, this.EndDate, this.IncludePatientTesting, this.IncludeUnitTesting, this.IncludeRackTesting, this.IncludeReagentTesting);

		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/11/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="5334"> 
		///		<ExpectedInput>Valid TestingWorklist object with all report section flags turned on</ExpectedInput>
		///		<ExpectedOutput>TestingWorklist report data generated</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5335"> 
		///		<ExpectedInput>Valid TestingWorklist object with all report section flags turned off</ExpectedInput>
		///		<ExpectedOutput>No TestingWorklist report data generated</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Creates testing worklist report based on criteria
		/// </summary>
		public void GetTestingWorklist()
		{
			if (this.IncludePatientTesting)
			{
				this.GetPatientTestingWorklist();
			}
			//
			if (this.IncludeUnitTesting)
			{
				this.GetBloodUnitTestingWorklist();
			}
			//
			if (this.IncludeRackTesting)
			{
				this.GetRackQCTestingWorklist();
			}
			//
			if (this.IncludeReagentTesting)
			{
				this.GetReagentQCTestingWorklist();
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/24/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4012"> 
		///		<ExpectedInput>Valid start and end dates in the past</ExpectedInput>
		///		<ExpectedOutput>Ordered test results data table</ExpectedOutput>
		///	</Case>
		///
		///	<Case type="0" testid ="4014"> 
		///		<ExpectedInput>Valid patient unique identifier and specimen list</ExpectedInput>
		///		<ExpectedOutput>Specimen test results data table</ExpectedOutput>
		///	</Case>
		///	
		/// <Case type="1" testid ="4013"> 
		///		<ExpectedInput>Invalid start and end dates in the future</ExpectedInput>
		///		<ExpectedOutput>Empty ordered test results data table</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Loads data table of patient testing information for a specific date range
		/// for use in UC_48 - Report Testing Worklist
		/// Implements BR_48.05, BR_48.06, BR_48.07, BR_48.08, BR_48.12, BR_48.13, BR_48.15, BR_77.05
		/// </summary>
		public void GetPatientTestingWorklist()
		{
			if ((this._patient == null)||(this._patient.PatientGuid == Guid.Empty))
			{
				//
				this._patientTestingWorklist = DAL.TestingWorklist.GetPatientTestingWorklist(this.StartDate,this.EndDate);
				//
			}
			else
			{
				DataTable dtTestWorklist = DAL.TestingWorklist.GetPatientTestingWorklist(this._patient.PatientGuid);
				//
				this._patientTestingWorklist = dtTestWorklist.Clone();
				//
				foreach(DataRow dr in dtTestWorklist.Rows)
				{
					foreach(BOL.Specimen checkSpecimen in this._specimens)
					{
						if (!dr.IsNull(TABLE.PatientSpecimen.PatientSpecimenGuid))
						{
							if (checkSpecimen.SpecimenGuid == (Guid)dr[TABLE.PatientSpecimen.PatientSpecimenGuid])
							{
								this._patientTestingWorklist.Rows.Add(dr.ItemArray);
								break;
							}
						}
					}
				}
			}
			//
			this._patientTestingWorklist.Columns.Add(ARTIFICIAL.GroupByPatient,typeof(string));
			this._patientTestingWorklist.Columns.Add(ARTIFICIAL.GroupByPatientTest,typeof(string));
			this._patientTestingWorklist.Columns.Add(Common.DatabaseConstants.ArtificialColumnNames.PatientDisplayName,typeof(string));
			this._patientTestingWorklist.Columns.Add(ARTIFICIAL.InvalidatedBy,typeof(string));
			this._patientTestingWorklist.Columns.Add(ARTIFICIAL.NoDataIndicator,typeof(bool));
			this._patientTestingWorklist.Columns.Add(ARTIFICIAL.Preliminary,typeof(bool));
			this._patientTestingWorklist.Columns[TABLE.SpecimenTest.TestComments].MaxLength = 355;
			//
			bool prelim = (this.EndDate.Date == BOL.VBECSDateTime.GetDivisionCurrentDateTime().Date);
			System.Text.StringBuilder comments = new System.Text.StringBuilder();;
			string previousGroupByTest = "";
			//
			foreach(DataRow dr in this._patientTestingWorklist.Rows)
			{
				dr[Common.DatabaseConstants.ArtificialColumnNames.PatientDisplayName] = BOL.Patient.BuildDisplayName(dr[TABLE.Patient.PatientFirstName].ToString(),dr[TABLE.Patient.PatientMiddleName].ToString(),dr[TABLE.Patient.PatientLastName].ToString());
				dr[ARTIFICIAL.GroupByPatient] = dr[TABLE.PatientSpecimen.PatientSpecimenGuid].ToString();
				if (!dr.IsNull(ARTIFICIAL.WorklistDate))
				{
					dr[ARTIFICIAL.GroupByPatientTest] = Convert.ToDateTime(dr[ARTIFICIAL.WorklistDate]).ToString("yyyyMMddHHmm");
				}
				else
				{
					dr[ARTIFICIAL.GroupByPatientTest] = Convert.ToDateTime(dr[TABLE.BloodUnit.LastUpdateDate]).ToString("yyyyMMddHHmm");
				}
				dr[ARTIFICIAL.GroupByPatientTest] += dr[ARTIFICIAL.TestGuid].ToString();
				dr[ARTIFICIAL.GroupByPatientTest] += dr[TABLE.ExceptionUnitSelection.UnitId].ToString();
				dr[ARTIFICIAL.GroupByPatientTest] += dr[TABLE.OrderableTest.OrderableTestText].ToString();
				dr[ARTIFICIAL.GroupByPatientTest] += dr[TABLE.BloodUnit.RecordStatusCode].ToString();
				if (!dr.IsNull(TABLE.SpecimenTest.InactivateDate))
				{
					dr[ARTIFICIAL.GroupByPatientTest] += Convert.ToDateTime(dr[TABLE.SpecimenTest.InactivateDate]).ToString("yyyyMMddHHmm");
				}
				//
				if (dr[ARTIFICIAL.GroupByPatientTest].ToString() != previousGroupByTest)
				{
					previousGroupByTest = dr[ARTIFICIAL.GroupByPatientTest].ToString();
					comments  = new System.Text.StringBuilder();
				}
				//
				if ((dr[TABLE.CannedComment.CannedCommentText].ToString().Length>0) && (comments.ToString().IndexOf(dr[TABLE.CannedComment.CannedCommentText].ToString())<0))
				{
					comments.Append(dr[TABLE.CannedComment.CannedCommentText].ToString()+"\n");
				}
				if ((dr[TABLE.BloodUnitTest.TestComments].ToString().Length>0) && (comments.ToString().IndexOf(dr[TABLE.BloodUnitTest.TestComments].ToString())<0))
				{
					comments.Append(dr[TABLE.BloodUnitTest.TestComments].ToString()+"\n");
				}
				if (comments.ToString().Length>255)
				{
					dr[TABLE.BloodUnitTest.TestComments] = comments.ToString().Substring(0,255);
				}
				else
				{
					dr[TABLE.BloodUnitTest.TestComments] = comments.ToString();
				}
				//
				if (Convert.ToChar(dr[TABLE.BloodUnit.RecordStatusCode]) != 'A')
				{
					dr[ARTIFICIAL.InvalidatedBy] = dr[TABLE.SpecimenTest.InactivateUser].ToString();
					if (!dr.IsNull(TABLE.SpecimenTest.InactivateDate))
					{
						dr[ARTIFICIAL.InvalidatedBy] += " on " + Common.VBECSDateTime.FormatDateTimeString(dr[TABLE.SpecimenTest.InactivateDate]);
					}
				}
				//
				if (((Convert.ToInt32(dr[TABLE.OrderableTest.OrderableTestId])==Convert.ToInt32(Common.OrderableTest.ABID))||
					(Convert.ToInt32(dr[TABLE.OrderableTest.OrderableTestId])==Convert.ToInt32(Common.OrderableTest.RepeatABID)))&&
					(dr.IsNull(TABLE.AntibodyType.AntibodyTypeName)))
				{
					dr[TABLE.AntibodyType.AntibodyTypeName] = "None identified";
				}
				//
				dr[ARTIFICIAL.NoDataIndicator] = false;
				dr[ARTIFICIAL.Preliminary] = prelim;
			}
			//
			if (this._specimens != null)
			{
				foreach(DataColumn col in this._patientTestingWorklist.Columns)
				{
					col.AllowDBNull = true;
				}
				//
				foreach(BOL.Specimen specimen in this._specimens)
				{
					DataRow[] drSpecimens = this._patientTestingWorklist.Select(TABLE.PatientSpecimen.PatientSpecimenGuid + " = '"+specimen.SpecimenGuid.ToString()+"'");
					//
					if ((drSpecimens == null)||(drSpecimens.Length==0))
					{
						DataRow drBlankRow = this._patientTestingWorklist.NewRow();
						//
						drBlankRow[ARTIFICIAL.GroupByPatient] = specimen.SpecimenGuid.ToString();
						drBlankRow[TABLE.PatientSpecimen.SpecimenAccessionNumber] = specimen.SpecimenUid;
						drBlankRow[TABLE.PatientSpecimen.SpecimenCollectionDate] = specimen.SpecimenCollectionDate;
						drBlankRow[Common.DatabaseConstants.ArtificialColumnNames.PatientDisplayName] = specimen.Patient.DisplayName;
						drBlankRow[TABLE.Patient.DisplayVistaPatientId] = specimen.Patient.DisplayPatientID;
						drBlankRow[ARTIFICIAL.NoDataIndicator] = true;
						this._patientTestingWorklist.Rows.Add(drBlankRow);
					}
				}
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/24/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4016"> 
		///		<ExpectedInput>Valid start and end dates in the past</ExpectedInput>
		///		<ExpectedOutput>Blood unit test data table</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4017"> 
		///		<ExpectedInput>Invalid start and end dates in the future</ExpectedInput>
		///		<ExpectedOutput>Empty blood unit test data table</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Loads data table of blood unit testing information
		/// for use in UC_48 - Report Testing Worklist
		/// Implements BR_48.07, BR_48.10, BR_48.13
		/// </summary>
		public void GetBloodUnitTestingWorklist()
		{
			this._unitTestingWorklist = DAL.TestingWorklist.GetBloodUnitTestingWorklist(this.StartDate,this.EndDate);
			this._unitTestingWorklist.Columns.Add(ARTIFICIAL.GroupBy,typeof(string));
			this._unitTestingWorklist.Columns.Add(ARTIFICIAL.InvalidatedBy,typeof(string));
			this._unitTestingWorklist.Columns.Add(ARTIFICIAL.Preliminary,typeof(bool));
			this._unitTestingWorklist.Columns[TABLE.BloodUnitTest.TestComments].MaxLength = 355;
			//
			bool prelim = (this.EndDate.Date == BOL.VBECSDateTime.GetDivisionCurrentDateTime().Date);
			//
			System.Text.StringBuilder comments = new System.Text.StringBuilder();;
			string previousGroupBy = "";
			//
			foreach(DataRow dr in this._unitTestingWorklist.Rows)
			{
				if (!dr.IsNull(ARTIFICIAL.WorklistDate))
				{
					dr[ARTIFICIAL.GroupBy] =  Convert.ToDateTime(dr[ARTIFICIAL.WorklistDate]).ToString("yyyyMMddHHmm");
				}
				else
				{
					dr[ARTIFICIAL.GroupBy] =  Convert.ToDateTime(dr[TABLE.BloodUnit.LastUpdateDate]).ToString("yyyyMMddHHmm");
				}
				dr[ARTIFICIAL.GroupBy] += dr[TABLE.BloodUnit.BloodUnitGuid].ToString();
				dr[ARTIFICIAL.GroupBy] += dr[ARTIFICIAL.BloodUnitTestId].ToString();
				dr[ARTIFICIAL.GroupBy] += dr[TABLE.BloodUnit.RecordStatusCode].ToString();
				if (!dr.IsNull(TABLE.SpecimenTest.InactivateDate))
				{
					dr[ARTIFICIAL.GroupBy] +=  Convert.ToDateTime(dr[TABLE.SpecimenTest.InactivateDate]).ToString("yyyyMMddHHmm");
				}
				//
				if (dr[ARTIFICIAL.GroupBy].ToString() != previousGroupBy)
				{
					previousGroupBy = dr[ARTIFICIAL.GroupBy].ToString();
					comments  = new System.Text.StringBuilder();
				}
				//
				this._unitTestingWorklist.Columns[TABLE.BloodUnitTest.TestComments].MaxLength = 355;
				if ((dr[TABLE.BloodUnitTest.TestComments].ToString().Length>0) && (comments.ToString().IndexOf(dr[TABLE.BloodUnitTest.TestComments].ToString())<0))
				{
					comments.Append(dr[TABLE.BloodUnitTest.TestComments].ToString()+"\n");
				}
				dr[TABLE.BloodUnitTest.TestComments] = comments.ToString();
				//
				if (Convert.ToChar(dr[TABLE.BloodUnit.RecordStatusCode]) != 'A')
				{
					dr[ARTIFICIAL.InvalidatedBy] = dr[TABLE.BloodUnitTest.InactivateUser].ToString();
					if (!dr.IsNull(TABLE.SpecimenTest.InactivateDate))
					{
						dr[ARTIFICIAL.InvalidatedBy] += " on " + Common.VBECSDateTime.FormatDateTimeString(dr[TABLE.SpecimenTest.InactivateDate]);
					}
				}
				//
				dr[ARTIFICIAL.Preliminary] = prelim;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/24/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4023"> 
		///		<ExpectedInput>Valid start and end dates in the past</ExpectedInput>
		///		<ExpectedOutput>Rack test results data table</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4024"> 
		///		<ExpectedInput>Invalid start and end dates in the future</ExpectedInput>
		///		<ExpectedOutput>Empty rack test results data table</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Loads a data table of daily QC testing information
		/// for use in UC_48 - Report Testing Worklist
		/// Implements BR_48.07, BR_48.11
		/// </summary>
		public void GetRackQCTestingWorklist()
		{
			this._rackTestingWorklist = DAL.TestingWorklist.GetRackQCTestingWorklist(this.StartDate,this.EndDate);
			this._rackTestingWorklist.Columns.Add(ARTIFICIAL.GroupBy,typeof(string));
			this._rackTestingWorklist.Columns.Add(ARTIFICIAL.InvalidatedBy,typeof(string));
			this._rackTestingWorklist.Columns.Add(ARTIFICIAL.Preliminary,typeof(bool));
			this._rackTestingWorklist.Columns[TABLE.RackLot.Comments].MaxLength = 355;
			//
			bool prelim = (this.EndDate.Date == BOL.VBECSDateTime.GetDivisionCurrentDateTime().Date);
			//
			//Create string column from RackGuid and TestDate, for use in grouping
			foreach(DataRow dr in this._rackTestingWorklist.Rows)
			{
				dr[ARTIFICIAL.GroupBy] = dr[TABLE.Rack.RackName].ToString() + "-" + Convert.ToDateTime(dr[TABLE.RackLot.LotUseDate]).ToString("yyyyMMdd") + "-" + dr[TABLE.BloodUnit.RecordStatusCode].ToString();
				//
				if (Convert.ToChar(dr[TABLE.BloodUnit.RecordStatusCode]) != 'A')
				{
					dr[ARTIFICIAL.InvalidatedBy] = dr[ARTIFICIAL.InvalidateUserName].ToString() + " on " + Common.VBECSDateTime.FormatDateTimeString(dr[TABLE.BloodUnit.LastUpdateDate]);
					dr[ARTIFICIAL.GroupBy] += "-" + Convert.ToDateTime(dr[TABLE.RackLot.LotUseDate]).ToString("yyyyMMddHHmmss");
				}
				//
				dr[ARTIFICIAL.Preliminary] = prelim;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/24/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4021"> 
		///		<ExpectedInput>Valid start and end dates in the past</ExpectedInput>
		///		<ExpectedOutput>Rack test results data table</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4022"> 
		///		<ExpectedInput>Invalid start and end dates in the future</ExpectedInput>
		///		<ExpectedOutput>Empty rack test results data table</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns a data table of non-daily QC reagent testing information
		/// for use in UC_48 - Report Testing Worklist
		/// Implements BR_48.07, BR_48.11
		/// </summary>
		public void GetReagentQCTestingWorklist()
		{
			this._reagentTestingWorklist = DAL.TestingWorklist.GetReagentQCTestingWorklist(this.StartDate,this.EndDate);
			this._reagentTestingWorklist.Columns.Add(ARTIFICIAL.GroupBy,typeof(string));
			this._reagentTestingWorklist.Columns.Add(ARTIFICIAL.Interp,typeof(string));
			this._reagentTestingWorklist.Columns.Add(ARTIFICIAL.InvalidatedBy,typeof(string));
			this._reagentTestingWorklist.Columns.Add(ARTIFICIAL.Preliminary,typeof(bool));
			this._reagentTestingWorklist.Columns[TABLE.RackLot.Comments].MaxLength = 355;
			//
			bool prelim = (this.EndDate.Date == BOL.VBECSDateTime.GetDivisionCurrentDateTime().Date);
			//
			foreach(DataRow dr in this._reagentTestingWorklist.Rows)
			{
				dr[ARTIFICIAL.GroupBy] = dr[TABLE.ReagentType.ReagentTypeName].ToString();
				dr[ARTIFICIAL.GroupBy] += dr[TABLE.Reagent.LotNumber].ToString();
				if (!dr.IsNull(ARTIFICIAL.WorklistDate))
				{
					dr[ARTIFICIAL.GroupBy] += Convert.ToDateTime(dr[ARTIFICIAL.WorklistDate]).ToString("yyyyMMddHHmm");
				}
				else
				{
					dr[ARTIFICIAL.GroupBy] += Convert.ToDateTime(dr[TABLE.BloodUnit.LastUpdateDate]).ToString("yyyyMMddHHmm");
				}
				dr[ARTIFICIAL.GroupBy] += dr[TABLE.RackLot.ControlCellId].ToString();
				dr[ARTIFICIAL.GroupBy] += dr[TABLE.BloodUnit.RecordStatusCode].ToString();
				dr[ARTIFICIAL.GroupBy] += dr[TABLE.RackLot.RackLotGuid].ToString();
				//
				if (!dr.IsNull(TABLE.RackLot.SatisfactoryResultIndicator))
				{
					if ((bool)dr[TABLE.RackLot.ControlCellId])
					{
						//Pos Control
						if ((bool)dr[TABLE.RackLot.SatisfactoryResultIndicator])
						{
							dr[ARTIFICIAL.Interp] = "Pos";
						}
						else
						{
							dr[ARTIFICIAL.Interp] = "Inc";
						}
					}
					else
					{
						//Neg Control
						if ((bool)dr[TABLE.RackLot.SatisfactoryResultIndicator])
						{
							dr[ARTIFICIAL.Interp] = "Neg";
						}
						else
						{
							dr[ARTIFICIAL.Interp] = "Inc";
						}
					}
				}
				else
				{
					dr[ARTIFICIAL.Interp] = "";
				}

				//
				if (Convert.ToChar(dr[TABLE.BloodUnit.RecordStatusCode]) != 'A')
				{
					dr[ARTIFICIAL.InvalidatedBy] = dr[ARTIFICIAL.InvalidateUserName].ToString() + " on " + Common.VBECSDateTime.FormatDateTimeString(dr[TABLE.BloodUnit.LastUpdateDate]);
				}
				//
				dr[ARTIFICIAL.Preliminary] = prelim;
			}
		}
/*
		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/15/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7827"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7828"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GenerateReport
		/// </summary>
		/// <param name="printedBy"></param>
		/// <returns></returns>
		public gov.va.med.vbecs.reports.TestingWorklist.TestingWorklistMain GenerateReport(string printedBy)
		{
			this.GetTestingWorklist();
			//
#if DEBUG
			//Un-comment lines if you need to recreate Xml schema
//			DataSet testingWorklist = new DataSet();
//			testingWorklist.Tables.Add(this.ReportSectionsTable.Copy());
//			testingWorklist.Tables.Add(this.PatientTestingWorklist.Copy());
//			testingWorklist.Tables.Add(this.UnitTestingWorklist.Copy());
//			testingWorklist.Tables.Add(this.RackTestingWorklist.Copy());
//			testingWorklist.Tables.Add(this.ReagentTestingWorklist.Copy());
//			testingWorklist.WriteXml("d:\\temp\\TestingWorklistReport.xml", XmlWriteMode.WriteSchema);
#endif
			//
			gov.va.med.vbecs.reports.TestingWorklist.TestingWorklistMain _rptTestingWorklistReport = new gov.va.med.vbecs.reports.TestingWorklist.TestingWorklistMain();
			_rptTestingWorklistReport.SetDataSource(this.ReportSectionsTable);
			//
			DataSet dsHeader =  BOL.VbecsReport.GetReportHeader(printedBy);
			_rptTestingWorklistReport.OpenSubreport("ReportHeader.rpt").SetDataSource(dsHeader);
			//
			DataSet dsFooter = BOL.VbecsReport.GetReportFooter(this.ReportCriteria);
			_rptTestingWorklistReport.OpenSubreport("ReportFooter.rpt").SetDataSource(dsFooter);
			//
			if (this.IncludePatientTesting)
			{
				_rptTestingWorklistReport.OpenSubreport("PatientTestingWorklist.rpt").SetDataSource(this.PatientTestingWorklist);
			}
			//
			if (this.IncludeUnitTesting)
			{
				_rptTestingWorklistReport.OpenSubreport("UnitTestingWorklist.rpt").SetDataSource(this.UnitTestingWorklist);
			}
			//
			if (this.IncludeRackTesting)
			{
				_rptTestingWorklistReport.OpenSubreport("RackTestingWorklist.rpt").SetDataSource(this.RackTestingWorklist);
			}
			//
			if (this.IncludeReagentTesting)
			{
				_rptTestingWorklistReport.OpenSubreport("ReagentTestingWorklist.rpt").SetDataSource(this.ReagentTestingWorklist);
			}
			//
			return _rptTestingWorklistReport;
		}
        */
		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/15/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7829"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7830"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Generate and print report
		/// Used by VbecsConsoleReport application
		/// </summary>
        protected override VbecsReportPrinter createReportPrinter()
		{
            this.GetTestingWorklist();
            BOL.VbecsUser user = new BOL.VbecsUser(this.PrintedBy);

            DataSet dsHeader = BOL.VbecsReport.GetReportHeader(user.UserName);
            DataSet dsFooter = BOL.VbecsReport.GetReportFooter(this.ReportCriteria);

            ReportDataSource reportSourceMain = new ReportDataSource();
            reportSourceMain.Name = "TestingWorklistDataSet";
            reportSourceMain.Value = this.ReportSectionsTable;

            ReportDataSource reportSourceHeader = new ReportDataSource();
            if (dsHeader.Tables.Count > 0)
            {
                reportSourceHeader.Name = "ReportHeaderDataSet";
                reportSourceHeader.Value = dsHeader.Tables[0];
            }

            ReportDataSource reportSourceFooter = new ReportDataSource();
            if (dsFooter.Tables.Count > 0)
            {
                reportSourceFooter.Name = "ReportFooterDataSet";
                reportSourceFooter.Value = dsFooter.Tables[0];
            }

            Assembly assembly = Assembly.LoadFrom(Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location) + "\\Reports.dll");
            Stream stream = assembly.GetManifestResourceStream("gov.va.med.vbecs.reports.TestingWorklist.TestingWorklistReport.rdlc");
            Stream streamPatient = assembly.GetManifestResourceStream("gov.va.med.vbecs.reports.TestingWorklist.PatientTestingWorklist.rdlc");
            Stream streamUnit = assembly.GetManifestResourceStream("gov.va.med.vbecs.reports.TestingWorklist.UnitTestingWorklist.rdlc");
            Stream streamRack = assembly.GetManifestResourceStream("gov.va.med.vbecs.reports.TestingWorklist.RackTestingWorklist.rdlc");
            Stream streamReagent = assembly.GetManifestResourceStream("gov.va.med.vbecs.reports.TestingWorklist.ReagentTestingWorklist.rdlc");

            LocalReport report = new LocalReport();
            report.SubreportProcessing += new SubreportProcessingEventHandler(LoadSubreportsEventHandler);

            report.DataSources.Add(reportSourceHeader);
            report.DataSources.Add(reportSourceFooter);
            report.DataSources.Add(reportSourceMain);
            report.LoadReportDefinition(stream);
            report.LoadSubreportDefinition("PatientTestingWorklist", streamPatient);
            report.LoadSubreportDefinition("UnitTestingWorklist", streamUnit);
            report.LoadSubreportDefinition("RackTestingWorklist", streamRack);
            report.LoadSubreportDefinition("ReagentTestingWorklist", streamReagent);

            return new BOL.VbecsReportPrinter(this.PrinterName, true, report);
        }

        private void LoadSubreportsEventHandler(object sender, SubreportProcessingEventArgs e)
        {
            if (e.ReportPath == "PatientTestingWorklist" || e.DataSourceNames[0].ToString() == "PatientTestingWorklistDataSet")
            {
                ReportDataSource reportPatient = new ReportDataSource();
                if (this.IncludePatientTesting)
                {
                    reportPatient.Name = "PatientTestingWorklistDataSet";
                    reportPatient.Value = this.PatientTestingWorklist;
                }
                e.DataSources.Add(reportPatient);
            }
            else if (e.ReportPath == "UnitTestingWorklist" || e.DataSourceNames[0].ToString() == "UnitTestingWorklistDataSet")
            {
                ReportDataSource reportUnit = new ReportDataSource();
                if (this.IncludeUnitTesting)
                {
                    reportUnit.Name = "UnitTestingWorklistDataSet";
                    reportUnit.Value = this.UnitTestingWorklist;
                }
                e.DataSources.Add(reportUnit);
            }
            else if (e.ReportPath == "RackTestingWorklist" || e.DataSourceNames[0].ToString() == "RackTestingWorklistDataSet")
            {
                ReportDataSource reportRack = new ReportDataSource();
                if (this.IncludeRackTesting)
                {
                    reportRack.Name = "RackTestingWorklistDataSet";
                    reportRack.Value = this.RackTestingWorklist;
                }
                e.DataSources.Add(reportRack);
            }
            else if (e.ReportPath == "ReagentTestingWorklist" || e.DataSourceNames[0].ToString() == "ReagentTestingWorklistDataSet")
            {
                ReportDataSource reportReagent = new ReportDataSource();
                if (this.IncludeReagentTesting)
                {
                    reportReagent.Name = "ReagentTestingWorklistDataSet";
                    reportReagent.Value = this.ReagentTestingWorklist;
                }
                e.DataSources.Add(reportReagent);
            }
        }
	}
}
